#!/usr/bin/env python3
"""
Analyze SAR log CSV files for high resource usage and network congestion indicators.
This script should be run from the CSV folder containing all the converted CSV files.
"""

import csv
import sys
from pathlib import Path
from collections import defaultdict
from datetime import datetime


def parse_timestamp(timestamp_str):
    """
    Parse timestamp string to datetime object for comparison.
    Format: "HH:MM:SS AM/PM"
    """
    try:
        # Parse format like "11:08:06 AM"
        time_part, am_pm = timestamp_str.rsplit(' ', 1)
        hour, minute, second = map(int, time_part.split(':'))
        if am_pm == 'PM' and hour != 12:
            hour += 12
        elif am_pm == 'AM' and hour == 12:
            hour = 0
        return hour * 3600 + minute * 60 + second  # Return seconds since midnight for comparison
    except:
        return 0


def read_csv_file(csv_path):
    """Read CSV file and return list of dictionaries."""
    data = []
    try:
        with open(csv_path, 'r', encoding='utf-8') as f:
            reader = csv.DictReader(f)
            for row in reader:
                data.append(row)
        return data
    except Exception as e:
        print(f"read_csv_file(): Error reading {csv_path}: {e}")
        return []


def check_cpu_usage(csv_dir):
    """Check CPU usage metrics."""
    issues = []
    csv_path = csv_dir / 'cpu_usage.csv'
    source_file = 'cpu_usage.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            user_pct = float(row.get('user_pct', 0))
            system_pct = float(row.get('system_pct', 0))
            iowait_pct = float(row.get('iowait_pct', 0))
            idle_pct = float(row.get('idle_pct', 100))
            steal_pct = float(row.get('steal_pct', 0))
            
            if user_pct > 80:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'CPU',
                    'metric': 'user_pct',
                    'value': user_pct,
                    'threshold': 80,
                    'message': f"High CPU user time: {user_pct:.2f}%",
                    'source_file': source_file
                })
            
            if system_pct > 20:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'CPU',
                    'metric': 'system_pct',
                    'value': system_pct,
                    'threshold': 20,
                    'message': f"High CPU system time: {system_pct:.2f}%",
                    'source_file': source_file
                })
            
            if iowait_pct > 10:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'CPU',
                    'metric': 'iowait_pct',
                    'value': iowait_pct,
                    'threshold': 10,
                    'message': f"High CPU I/O wait: {iowait_pct:.2f}% (possible disk bottleneck)",
                    'source_file': source_file
                })
            
            if idle_pct < 20:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'CPU',
                    'metric': 'idle_pct',
                    'value': idle_pct,
                    'threshold': 20,
                    'message': f"Low CPU idle: {idle_pct:.2f}%",
                    'source_file': source_file
                })
            
            if steal_pct > 5:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'CPU',
                    'metric': 'steal_pct',
                    'value': steal_pct,
                    'threshold': 5,
                    'message': f"High CPU steal time: {steal_pct:.2f}% (virtualization overhead)",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_per_cpu_usage(csv_dir, num_cores=24):
    """Check per-CPU usage for individual core issues."""
    issues = []
    csv_path = csv_dir / 'per_cpu.csv'
    source_file = 'per_cpu.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            # Check each CPU core
            for cpu_num in range(num_cores):
                cpu_prefix = f'cpu{cpu_num}_'
                user_pct = float(row.get(f'{cpu_prefix}user_pct', 0))
                idle_pct = float(row.get(f'{cpu_prefix}idle_pct', 100))
                
                if user_pct > 90:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'CPU',
                        'metric': f'cpu{cpu_num}_user_pct',
                        'value': user_pct,
                        'threshold': 90,
                        'message': f"CPU core {cpu_num} high usage: {user_pct:.2f}%",
                        'source_file': source_file
                    })
                
                if idle_pct < 10:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'CPU',
                        'metric': f'cpu{cpu_num}_idle_pct',
                        'value': idle_pct,
                        'threshold': 10,
                        'message': f"CPU core {cpu_num} low idle: {idle_pct:.2f}%",
                        'source_file': source_file
                    })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_memory_usage(csv_dir):
    """Check memory usage metrics."""
    issues = []
    csv_path = csv_dir / 'mem_usage.csv'
    source_file = 'mem_usage.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            # memused_pct = float(row.get('memused_pct', 0))
            commit_pct = float(row.get('commit_pct', 0))
            
            # if memused_pct > 95:
            #     issues.append({
            #         'timestamp': timestamp,
            #         'severity': 'CRITICAL',
            #         'category': 'Memory',
            #         'metric': 'memused_pct',
            #         'value': memused_pct,
            #         'threshold': 95,
            #         'message': f"Critical memory usage: {memused_pct:.2f}%",
            #         'source_file': source_file
            #     })
            # elif memused_pct > 85:
            #     issues.append({
            #         'timestamp': timestamp,
            #         'severity': 'WARNING',
            #         'category': 'Memory',
            #         'metric': 'memused_pct',
            #         'value': memused_pct,
            #         'threshold': 85,
            #         'message': f"High memory usage: {memused_pct:.2f}%",
            #         'source_file': source_file
            #     })
            
            if commit_pct > 95:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'Memory',
                    'metric': 'commit_pct',
                    'value': commit_pct,
                    'threshold': 95,
                    'message': f"Critical memory commit: {commit_pct:.2f}% (risk of OOM)",
                    'source_file': source_file
                })
            elif commit_pct > 85:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Memory',
                    'metric': 'commit_pct',
                    'value': commit_pct,
                    'threshold': 85,
                    'message': f"High memory commit: {commit_pct:.2f}% (risk of OOM)",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_swap_usage(csv_dir):
    """Check swap usage metrics."""
    issues = []
    csv_path = csv_dir / 'swap_usage.csv'
    source_file = 'swap_usage.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            swpused_pct = float(row.get('swpused_pct', 0))
            
            if swpused_pct > 20:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'Memory',
                    'metric': 'swpused_pct',
                    'value': swpused_pct,
                    'threshold': 20,
                    'message': f"Critical swap usage: {swpused_pct:.2f}%",
                    'source_file': source_file
                })
            elif swpused_pct > 10:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Memory',
                    'metric': 'swpused_pct',
                    'value': swpused_pct,
                    'threshold': 10,
                    'message': f"High swap usage: {swpused_pct:.2f}%",
                    'source_file': 'swap_usage.csv'
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_load_average(csv_dir, num_cores=24):
    """Check load average metrics."""
    issues = []
    csv_path = csv_dir / 'load_avg.csv'
    source_file = 'load_avg.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            ldavg_1 = float(row.get('ldavg_1', 0))
            ldavg_5 = float(row.get('ldavg_5', 0))
            ldavg_15 = float(row.get('ldavg_15', 0))
            blocked = int(row.get('blocked', 0))
            
            threshold_1 = num_cores * 1.5
            threshold_5 = num_cores * 1.2
            threshold_15 = num_cores
            
            if ldavg_1 > threshold_1:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'Load',
                    'metric': 'ldavg_1',
                    'value': ldavg_1,
                    'threshold': threshold_1,
                    'message': f"Critical 1-minute load average: {ldavg_1:.2f} (threshold: {threshold_1:.1f})",
                    'source_file': source_file
                })
            elif ldavg_1 > num_cores:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Load',
                    'metric': 'ldavg_1',
                    'value': ldavg_1,
                    'threshold': num_cores,
                    'message': f"High 1-minute load average: {ldavg_1:.2f} (CPUs: {num_cores})",
                    'source_file': source_file
                })
            
            if ldavg_5 > threshold_5:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Load',
                    'metric': 'ldavg_5',
                    'value': ldavg_5,
                    'threshold': threshold_5,
                    'message': f"High 5-minute load average: {ldavg_5:.2f}",
                    'source_file': source_file
                })
            
            if ldavg_15 > threshold_15:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Load',
                    'metric': 'ldavg_15',
                    'value': ldavg_15,
                    'threshold': threshold_15,
                    'message': f"High 15-minute load average: {ldavg_15:.2f}",
                    'source_file': source_file
                })
            
            if blocked > 0:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Load',
                    'metric': 'blocked',
                    'value': blocked,
                    'threshold': 0,
                    'message': f"Processes blocked on I/O: {blocked}",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_disk_io(csv_dir):
    """Check disk I/O metrics."""
    issues = []
    csv_path = csv_dir / 'disk_io.csv'
    source_file = 'disk_io.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    # Get all device columns
    if not data:
        return issues
    
    device_metrics = {}
    for key in data[0].keys():
        if key != 'timestamp' and '_' in key:
            parts = key.rsplit('_', 1)
            if len(parts) == 2:
                device = parts[0]
                metric = parts[1]
                if device not in device_metrics:
                    device_metrics[device] = []
                device_metrics[device].append(metric)
    
    for row in data:
        timestamp = row.get('timestamp', '')
        for device, metrics in device_metrics.items():
            try:
                util_pct = float(row.get(f'{device}_util_pct', 0))
                await_val = float(row.get(f'{device}_await', 0))
                avgqu_sz = float(row.get(f'{device}_avgqu_sz', 0))
                tps = float(row.get(f'{device}_tps', 0))
                
                if util_pct > 90:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'CRITICAL',
                        'category': 'Disk I/O',
                        'metric': f'{device}_util_pct',
                        'value': util_pct,
                        'threshold': 90,
                        'message': f"Critical disk utilization on {device}: {util_pct:.2f}%",
                        'source_file': source_file
                    })
                elif util_pct > 80:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Disk I/O',
                        'metric': f'{device}_util_pct',
                        'value': util_pct,
                        'threshold': 80,
                        'message': f"High disk utilization on {device}: {util_pct:.2f}%",
                        'source_file': source_file
                    })
                
                if await_val > 50:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Disk I/O',
                        'metric': f'{device}_await',
                        'value': await_val,
                        'threshold': 50,
                        'message': f"High I/O wait time on {device}: {await_val:.2f}ms",
                        'source_file': source_file
                    })
                
                if avgqu_sz > 10:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Disk I/O',
                        'metric': f'{device}_avgqu_sz',
                        'value': avgqu_sz,
                        'threshold': 10,
                        'message': f"Large I/O queue on {device}: {avgqu_sz:.2f}",
                        'source_file': source_file
                    })
                
                if tps > 1000:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'INFO',
                        'category': 'Disk I/O',
                        'metric': f'{device}_tps',
                        'value': tps,
                        'threshold': 1000,
                        'message': f"High disk transactions on {device}: {tps:.2f}/s",
                        'source_file': source_file
                    })
            except (ValueError, KeyError):
                continue
    
    return issues


def check_network_devices(csv_dir):
    """Check network device metrics."""
    issues = []
    csv_path = csv_dir / 'net_dev.csv'
    source_file = 'net_dev.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    # Get all device columns
    device_metrics = {}
    for key in data[0].keys():
        if key != 'timestamp' and '_' in key:
            parts = key.rsplit('_', 1)
            if len(parts) == 2:
                device = parts[0]
                metric = parts[1]
                if device not in device_metrics:
                    device_metrics[device] = []
                device_metrics[device].append(metric)
    
    for row in data:
        timestamp = row.get('timestamp', '')
        for device, metrics in device_metrics.items():
            try:
                rxpck_s = float(row.get(f'{device}_rxpck_s', 0))
                txpck_s = float(row.get(f'{device}_txpck_s', 0))
                rxkB_s = float(row.get(f'{device}_rxkB_s', 0))
                txkB_s = float(row.get(f'{device}_txkB_s', 0))
                
                # High packet rates (threshold depends on interface, using conservative 100k)
                if rxpck_s > 100000 or txpck_s > 100000:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network',
                        'metric': f'{device}_pck_rate',
                        'value': max(rxpck_s, txpck_s),
                        'threshold': 100000,
                        'message': f"High packet rate on {device}: RX={rxpck_s:.0f}/s, TX={txpck_s:.0f}/s",
                        'source_file': source_file
                    })
                
                # High bandwidth (assuming 1Gbps interface, 80% = ~100 MB/s)
                if rxkB_s > 100000 or txkB_s > 100000:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network',
                        'metric': f'{device}_bandwidth',
                        'value': max(rxkB_s, txkB_s),
                        'threshold': 100000,
                        'message': f"High bandwidth on {device}: RX={rxkB_s:.2f} kB/s, TX={txkB_s:.2f} kB/s",
                        'source_file': source_file
                    })
            except (ValueError, KeyError):
                continue
    
    return issues


def check_network_errors(csv_dir):
    """Check network error metrics."""
    issues = []
    csv_path = csv_dir / 'net_errors.csv'
    source_file = 'net_errors.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    # Get all device columns
    device_metrics = {}
    for key in data[0].keys():
        if key != 'timestamp' and '_' in key:
            parts = key.rsplit('_', 1)
            if len(parts) == 2:
                device = parts[0]
                metric = parts[1]
                if device not in device_metrics:
                    device_metrics[device] = []
                device_metrics[device].append(metric)
    
    for row in data:
        timestamp = row.get('timestamp', '')
        for device, metrics in device_metrics.items():
            try:
                rxerr_s = float(row.get(f'{device}_rxerr_s', 0))
                txerr_s = float(row.get(f'{device}_txerr_s', 0))
                rxdrop_s = float(row.get(f'{device}_rxdrop_s', 0))
                txdrop_s = float(row.get(f'{device}_txdrop_s', 0))
                coll_s = float(row.get(f'{device}_coll_s', 0))
                rxfram_s = float(row.get(f'{device}_rxfram_s', 0))
                rxfifo_s = float(row.get(f'{device}_rxfifo_s', 0))
                txfifo_s = float(row.get(f'{device}_txfifo_s', 0))
                
                if rxerr_s > 100 or txerr_s > 100:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'CRITICAL',
                        'category': 'Network Errors',
                        'metric': f'{device}_errors',
                        'value': max(rxerr_s, txerr_s),
                        'threshold': 100,
                        'message': f"Network errors on {device}: RX={rxerr_s:.0f}/s, TX={txerr_s:.0f}/s",
                        'source_file': source_file
                    })
                elif rxerr_s > 0 or txerr_s > 0:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network Errors',
                        'metric': f'{device}_errors',
                        'value': max(rxerr_s, txerr_s),
                        'threshold': 0,
                        'message': f"Network errors detected on {device}: RX={rxerr_s:.0f}/s, TX={txerr_s:.0f}/s",
                        'source_file': source_file
                    })
                
                if rxdrop_s > 100 or txdrop_s > 100:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'CRITICAL',
                        'category': 'Network Errors',
                        'metric': f'{device}_drops',
                        'value': max(rxdrop_s, txdrop_s),
                        'threshold': 100,
                        'message': f"Packet drops on {device}: RX={rxdrop_s:.0f}/s, TX={txdrop_s:.0f}/s (congestion)",
                        'source_file': source_file
                    })
                elif rxdrop_s > 0 or txdrop_s > 0:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network Errors',
                        'metric': f'{device}_drops',
                        'value': max(rxdrop_s, txdrop_s),
                        'threshold': 0,
                        'message': f"Packet drops detected on {device}: RX={rxdrop_s:.0f}/s, TX={txdrop_s:.0f}/s",
                        'source_file': source_file
                    })
                
                if coll_s > 100:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network Errors',
                        'metric': f'{device}_collisions',
                        'value': coll_s,
                        'threshold': 100,
                        'message': f"High collisions on {device}: {coll_s:.0f}/s",
                        'source_file': source_file
                    })
                
                if rxfram_s > 0 or rxfifo_s > 0 or txfifo_s > 0:
                    issues.append({
                        'timestamp': timestamp,
                        'severity': 'WARNING',
                        'category': 'Network Errors',
                        'metric': f'{device}_fram_fifo',
                        'value': max(rxfram_s, rxfifo_s, txfifo_s),
                        'threshold': 0,
                        'message': f"Frame/FIFO errors on {device}: frames={rxfram_s:.0f}/s, RX FIFO={rxfifo_s:.0f}/s, TX FIFO={txfifo_s:.0f}/s",
                        'source_file': source_file
                    })
            except (ValueError, KeyError):
                continue
    
    return issues


def check_page_faults(csv_dir):
    """Check page fault metrics."""
    issues = []
    csv_path = csv_dir / 'page_faults.csv'
    source_file = 'page_faults.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            fault_s = float(row.get('fault_s', 0))
            majflt_s = float(row.get('majflt_s', 0))
            
            if majflt_s > 100:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'Memory',
                    'metric': 'majflt_s',
                    'value': majflt_s,
                    'threshold': 100,
                    'message': f"Critical major page faults: {majflt_s:.0f}/s (indicates swapping)",
                    'source_file': source_file
                })
            elif majflt_s > 10:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Memory',
                    'metric': 'majflt_s',
                    'value': majflt_s,
                    'threshold': 10,
                    'message': f"High major page faults: {majflt_s:.0f}/s",
                    'source_file': source_file
                })
            
            if fault_s > 100000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Memory',
                    'metric': 'fault_s',
                    'value': fault_s,
                    'threshold': 100000,
                    'message': f"Very high page fault rate: {fault_s:.0f}/s",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_paging(csv_dir):
    """Check swap paging activity."""
    issues = []
    csv_path = csv_dir / 'paging.csv'
    source_file = 'paging.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            pswpin_s = float(row.get('pswpin_s', 0))
            pswpout_s = float(row.get('pswpout_s', 0))
            
            if pswpin_s > 1000 or pswpout_s > 1000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'CRITICAL',
                    'category': 'Memory',
                    'metric': 'swap_activity',
                    'value': max(pswpin_s, pswpout_s),
                    'threshold': 1000,
                    'message': f"Critical swap activity: in={pswpin_s:.0f}/s, out={pswpout_s:.0f}/s",
                    'source_file': source_file
                })
            elif pswpin_s > 100 or pswpout_s > 100:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Memory',
                    'metric': 'swap_activity',
                    'value': max(pswpin_s, pswpout_s),
                    'threshold': 100,
                    'message': f"High swap activity: in={pswpin_s:.0f}/s, out={pswpout_s:.0f}/s",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_context_switches(csv_dir):
    """Check context switch metrics."""
    issues = []
    csv_path = csv_dir / 'context_switches.csv'
    source_file = 'context_switches.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            cswch_s = float(row.get('cswch_s', 0))
            proc_s = float(row.get('proc_s', 0))
            
            if cswch_s > 100000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'INFO',
                    'category': 'System',
                    'metric': 'cswch_s',
                    'value': cswch_s,
                    'threshold': 100000,
                    'message': f"Very high context switch rate: {cswch_s:.0f}/s",
                    'source_file': source_file
                })
            
            if proc_s > 100:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'System',
                    'metric': 'proc_s',
                    'value': proc_s,
                    'threshold': 100,
                    'message': f"High process creation rate: {proc_s:.0f}/s",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def check_socket_usage(csv_dir):
    """Check socket usage metrics."""
    issues = []
    csv_path = csv_dir / 'sock_usage.csv'
    source_file = 'sock_usage.csv'
    
    if not csv_path.exists():
        return issues
    
    data = read_csv_file(csv_path)
    if not data:
        return issues
    
    for row in data:
        timestamp = row.get('timestamp', '')
        try:
            totsck = int(row.get('totsck', 0))
            tcpsck = int(row.get('tcpsck', 0))
            tcp_tw = int(row.get('tcp_tw', 0))
            
            if totsck > 50000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Network',
                    'metric': 'totsck',
                    'value': totsck,
                    'threshold': 50000,
                    'message': f"High total socket count: {totsck}",
                    'source_file': source_file
                })
            
            if tcpsck > 30000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'WARNING',
                    'category': 'Network',
                    'metric': 'tcpsck',
                    'value': tcpsck,
                    'threshold': 30000,
                    'message': f"High TCP socket count: {tcpsck}",
                    'source_file': source_file
                })
            
            if tcp_tw > 10000:
                issues.append({
                    'timestamp': timestamp,
                    'severity': 'INFO',
                    'category': 'Network',
                    'metric': 'tcp_tw',
                    'value': tcp_tw,
                    'threshold': 10000,
                    'message': f"High TIME_WAIT sockets: {tcp_tw} (connection churn)",
                    'source_file': source_file
                })
        except (ValueError, KeyError):
            continue
    
    return issues


def analyze_all_csv_files(csv_dir='.'):
    """
    Analyze all CSV files for high resource usage and network congestion.
    
    Args:
        csv_dir: Directory containing CSV files (default: current directory)
    """
    csv_path = Path(csv_dir)
    
    if not csv_path.exists():
        print(f"analyze_all_csv_files(): Error - CSV directory not found: {csv_path.absolute()}")
        sys.exit(1)
    
    print("analyze_all_csv_files(): Analyzing SAR log CSV files for high resource usage indicators...")
    print(f"analyze_all_csv_files(): Reading CSV files from: {csv_path.absolute()}\n")
    
    all_issues = []
    
    # Run all checks
    print("analyze_all_csv_files(): Checking CPU usage...")
    all_issues.extend(check_cpu_usage(csv_path))
    
    print("analyze_all_csv_files(): Checking per-CPU usage...")
    all_issues.extend(check_per_cpu_usage(csv_path))
    
    print("analyze_all_csv_files(): Checking memory usage...")
    all_issues.extend(check_memory_usage(csv_path))
    
    print("analyze_all_csv_files(): Checking swap usage...")
    all_issues.extend(check_swap_usage(csv_path))
    
    print("analyze_all_csv_files(): Checking load average...")
    all_issues.extend(check_load_average(csv_path))
    
    print("analyze_all_csv_files(): Checking disk I/O...")
    all_issues.extend(check_disk_io(csv_path))
    
    print("analyze_all_csv_files(): Checking network devices...")
    all_issues.extend(check_network_devices(csv_path))
    
    print("analyze_all_csv_files(): Checking network errors...")
    all_issues.extend(check_network_errors(csv_path))
    
    print("analyze_all_csv_files(): Checking page faults...")
    all_issues.extend(check_page_faults(csv_path))
    
    print("analyze_all_csv_files(): Checking paging activity...")
    all_issues.extend(check_paging(csv_path))
    
    print("analyze_all_csv_files(): Checking context switches...")
    all_issues.extend(check_context_switches(csv_path))
    
    print("analyze_all_csv_files(): Checking socket usage...")
    all_issues.extend(check_socket_usage(csv_path))
    
    # Sort issues by severity and timestamp
    severity_order = {'CRITICAL': 0, 'WARNING': 1, 'INFO': 2}
    all_issues.sort(key=lambda x: (severity_order.get(x['severity'], 3), x.get('source_file', ''), x['timestamp']))
    
    # Write issues to file
    output_file = csv_path / 'resource_usage_issues.txt'
    with open(output_file, 'w', encoding='utf-8') as f:
        f.write("=" * 80 + "\n")
        f.write("RESOURCE USAGE ANALYSIS - DETAILED ISSUES\n")
        f.write("=" * 80 + "\n\n")
        
        # Group by severity
        by_severity = defaultdict(list)
        for issue in all_issues:
            by_severity[issue['severity']].append(issue)
        
        # Write issues grouped by severity
        for severity in ['CRITICAL', 'WARNING', 'INFO']:
            if by_severity[severity]:
                f.write(f"\n{severity} ISSUES ({len(by_severity[severity])}):\n")
                f.write("-" * 80 + "\n")
                for issue in by_severity[severity]:
                    source = issue.get('source_file', 'unknown')
                    f.write(f"  [{issue['timestamp']}] [{source}] {issue['category']}: {issue['message']}\n")
                    f.write(f"      Metric: {issue['metric']}, Value: {issue['value']:.2f}, Threshold: {issue['threshold']}\n")
        
        if not all_issues:
            f.write("\nNo high resource usage or network congestion indicators detected.\n")
            f.write("System appears to be operating within normal parameters.\n")
    
    # Group issues by source file for summary
    by_source_file = defaultdict(lambda: {'CRITICAL': 0, 'WARNING': 0, 'INFO': 0})
    for issue in all_issues:
        source = issue.get('source_file', 'unknown')
        by_source_file[source][issue['severity']] += 1
    
    # Print summary
    print(f"\n{'='*80}")
    print("ANALYSIS SUMMARY")
    print(f"{'='*80}")
    print(f"Total issues found: {len(all_issues)}")
    
    # Group by severity for overall summary
    by_severity = defaultdict(int)
    for issue in all_issues:
        by_severity[issue['severity']] += 1
    
    print(f"  CRITICAL: {by_severity['CRITICAL']}")
    print(f"  WARNING:  {by_severity['WARNING']}")
    print(f"  INFO:     {by_severity['INFO']}")
    print(f"{'='*80}\n")
    
    # Print summary by CSV file/log type
    print("ISSUES BY CSV FILE (LOG TYPE):")
    print("-" * 80)
    print(f"{'CSV File':<30} {'CRITICAL':<12} {'WARNING':<12}")
    print("-" * 80)
    
    # Sort by filename
    sorted_files = sorted(by_source_file.keys())
    for source_file in sorted_files:
        counts = by_source_file[source_file]
        critical_count = counts['CRITICAL']
        warning_count = counts['WARNING']
        if critical_count > 0 or warning_count > 0:
            print(f"{source_file:<30} {critical_count:<12} {warning_count:<12}")
    
    if not any(by_source_file[f]['CRITICAL'] > 0 or by_source_file[f]['WARNING'] > 0 for f in by_source_file):
        print("\nNo CRITICAL or WARNING issues found in any CSV files.")
    
    print(f"\nDetailed issues written to: {output_file.absolute()}")
    
    return all_issues


if __name__ == '__main__':
    csv_directory = sys.argv[1] if len(sys.argv) > 1 else '.'
    analyze_all_csv_files(csv_directory)