#!/usr/bin/env python3
"""
Convert swap_usage.log file to CSV format.
Each timestamp becomes one row in the CSV output.
"""

import csv
import re
import sys
from pathlib import Path


def parse_swap_usage_log(input_file, output_file):
    """
    Parse swap_usage.log file and convert to CSV format.
    
    Args:
        input_file: Path to the input swap_usage.log file
        output_file: Path to the output CSV file
    """
    input_path = Path(input_file)
    
    if not input_path.exists():
        print(f"parse_swap_usage_log(): Error - Input file not found: {input_file}")
        sys.exit(1)
    
    # CSV column headers
    csv_headers = ['timestamp', 'kbswpfree', 'kbswpused', 'swpused_pct', 'kbswpcad', 'swpcad_pct']
    
    with open(input_path, 'r', encoding='utf-8') as infile, \
         open(output_file, 'w', newline='', encoding='utf-8') as outfile:
        
        csv_writer = csv.writer(outfile)
        csv_writer.writerow(csv_headers)
        
        # Skip the first 3 lines (system info header, empty line, column header)
        for _ in range(3):
            infile.readline()
        
        # Process data rows
        line_count = 0
        for line in infile:
            line = line.strip()
            
            # Skip empty lines
            if not line:
                continue
            
            # Parse the line - format: "HH:MM:SS AM/PM    kbswpfree    kbswpused    %swpused    kbswpcad    %swpcad"
            # Use regex to split on multiple spaces
            parts = re.split(r'\s+', line)
            
            if len(parts) < 7:
                print(f"parse_swap_usage_log(): Warning - Skipping malformed line: {line}")
                continue
            
            timestamp = f"{parts[0]} {parts[1]}"  # Time and AM/PM
            kbswpfree = parts[2]
            kbswpused = parts[3]
            swpused_pct = parts[4]
            kbswpcad = parts[5]
            swpcad_pct = parts[6]
            
            # Write row to CSV
            csv_writer.writerow([timestamp, kbswpfree, kbswpused, swpused_pct, kbswpcad, swpcad_pct])
            line_count += 1
        
        print(f"parse_swap_usage_log(): Successfully converted {line_count} rows to CSV")


if __name__ == '__main__':
    # Default input and output paths
    default_input = 'sar_logs_20251112_110805/swap_usage.log'
    default_output = 'swap_usage.csv'
    
    # Allow command line arguments for flexibility
    input_file = sys.argv[1] if len(sys.argv) > 1 else default_input
    output_file = sys.argv[2] if len(sys.argv) > 2 else default_output
    
    parse_swap_usage_log(input_file, output_file)
    print(f"parse_swap_usage_log(): Output written to: {output_file}")

