#!/usr/bin/env python3
"""
Convert ip_stats.log file to CSV format.
Each timestamp becomes one row in the CSV output.
"""

import csv
import re
import sys
from pathlib import Path


def parse_ip_stats_log(input_file, output_file):
    """
    Parse ip_stats.log file and convert to CSV format.
    
    Args:
        input_file: Path to the input ip_stats.log file
        output_file: Path to the output CSV file
    """
    input_path = Path(input_file)
    
    if not input_path.exists():
        print(f"parse_ip_stats_log(): Error - Input file not found: {input_file}")
        sys.exit(1)
    
    # CSV column headers
    csv_headers = ['timestamp', 'irec_s', 'fwddgm_s', 'idel_s', 'orq_s', 'asmrq_s', 'asmok_s', 'fragok_s', 'fragcrt_s']
    
    with open(input_path, 'r', encoding='utf-8') as infile, \
         open(output_file, 'w', newline='', encoding='utf-8') as outfile:
        
        csv_writer = csv.writer(outfile)
        csv_writer.writerow(csv_headers)
        
        # Skip the first 3 lines (system info header, empty line, column header)
        for _ in range(3):
            infile.readline()
        
        # Process data rows
        line_count = 0
        for line in infile:
            line = line.strip()
            
            # Skip empty lines
            if not line:
                continue
            
            # Parse the line - format: "HH:MM:SS AM/PM    irec/s    fwddgm/s    idel/s    orq/s    asmrq/s    asmok/s    fragok/s    fragcrt/s"
            # Use regex to split on multiple spaces
            parts = re.split(r'\s+', line)
            
            if len(parts) < 9:
                print(f"parse_ip_stats_log(): Warning - Skipping malformed line: {line}")
                continue
            
            timestamp = f"{parts[0]} {parts[1]}"  # Time and AM/PM
            irec_s = parts[2]
            fwddgm_s = parts[3]
            idel_s = parts[4]
            orq_s = parts[5]
            asmrq_s = parts[6]
            asmok_s = parts[7]
            fragok_s = parts[8]
            fragcrt_s = parts[9] if len(parts) > 9 else '0.00'
            
            # Write row to CSV
            csv_writer.writerow([timestamp, irec_s, fwddgm_s, idel_s, orq_s, asmrq_s, asmok_s, fragok_s, fragcrt_s])
            line_count += 1
        
        print(f"parse_ip_stats_log(): Successfully converted {line_count} rows to CSV")


if __name__ == '__main__':
    # Default input and output paths
    default_input = 'sar_logs_20251112_110805/ip_stats.log'
    default_output = 'ip_stats.csv'
    
    # Allow command line arguments for flexibility
    input_file = sys.argv[1] if len(sys.argv) > 1 else default_input
    output_file = sys.argv[2] if len(sys.argv) > 2 else default_output
    
    parse_ip_stats_log(input_file, output_file)
    print(f"parse_ip_stats_log(): Output written to: {output_file}")

