#!/usr/bin/env python3
"""
Convert load_avg.log file to CSV format.
Each timestamp becomes one row in the CSV output.
"""

import csv
import re
import sys
from pathlib import Path


def parse_load_avg_log(input_file, output_file):
    """
    Parse load_avg.log file and convert to CSV format.
    
    Args:
        input_file: Path to the input load_avg.log file
        output_file: Path to the output CSV file
    """
    input_path = Path(input_file)
    
    if not input_path.exists():
        print(f"parse_load_avg_log(): Error - Input file not found: {input_file}")
        sys.exit(1)
    
    # CSV column headers
    csv_headers = ['timestamp', 'runq_sz', 'plist_sz', 'ldavg_1', 'ldavg_5', 'ldavg_15', 'blocked']
    
    with open(input_path, 'r', encoding='utf-8') as infile, \
         open(output_file, 'w', newline='', encoding='utf-8') as outfile:
        
        csv_writer = csv.writer(outfile)
        csv_writer.writerow(csv_headers)
        
        # Skip the first 3 lines (system info header, empty line, column header)
        for _ in range(3):
            infile.readline()
        
        # Process data rows
        line_count = 0
        for line in infile:
            line = line.strip()
            
            # Skip empty lines
            if not line:
                continue
            
            # Parse the line - format: "HH:MM:SS AM/PM    runq-sz    plist-sz    ldavg-1    ldavg-5    ldavg-15    blocked"
            # Use regex to split on multiple spaces
            parts = re.split(r'\s+', line)
            
            if len(parts) < 7:
                print(f"parse_load_avg_log(): Warning - Skipping malformed line: {line}")
                continue
            
            timestamp = f"{parts[0]} {parts[1]}"  # Time and AM/PM
            runq_sz = parts[2]
            plist_sz = parts[3]
            ldavg_1 = parts[4]
            ldavg_5 = parts[5]
            ldavg_15 = parts[6]
            blocked = parts[7] if len(parts) > 7 else '0'
            
            # Write row to CSV
            csv_writer.writerow([timestamp, runq_sz, plist_sz, ldavg_1, ldavg_5, ldavg_15, blocked])
            line_count += 1
        
        print(f"parse_load_avg_log(): Successfully converted {line_count} rows to CSV")


if __name__ == '__main__':
    # Default input and output paths
    default_input = 'sar_logs_20251112_110805/load_avg.log'
    default_output = 'load_avg.csv'
    
    # Allow command line arguments for flexibility
    input_file = sys.argv[1] if len(sys.argv) > 1 else default_input
    output_file = sys.argv[2] if len(sys.argv) > 2 else default_output
    
    parse_load_avg_log(input_file, output_file)
    print(f"parse_load_avg_log(): Output written to: {output_file}")

