#!/usr/bin/env python3
"""
Convert partition_io.log file to CSV format.
Note: This file appears to contain CPU usage data, not partition I/O data.
Each timestamp becomes one row in the CSV output.
"""

import csv
import re
import sys
from pathlib import Path


def parse_partition_io_log(input_file, output_file):
    """
    Parse partition_io.log file and convert to CSV format.
    Note: This file appears to contain CPU usage data.
    
    Args:
        input_file: Path to the input partition_io.log file
        output_file: Path to the output CSV file
    """
    input_path = Path(input_file)
    
    if not input_path.exists():
        print(f"parse_partition_io_log(): Error - Input file not found: {input_file}")
        sys.exit(1)
    
    # CSV column headers (CPU usage format)
    csv_headers = ['timestamp', 'cpu', 'user_pct', 'nice_pct', 'system_pct', 'iowait_pct', 'steal_pct', 'idle_pct']
    
    with open(input_path, 'r', encoding='utf-8') as infile, \
         open(output_file, 'w', newline='', encoding='utf-8') as outfile:
        
        csv_writer = csv.writer(outfile)
        csv_writer.writerow(csv_headers)
        
        # Skip the first 3 lines (system info header, empty line, column header)
        for _ in range(3):
            infile.readline()
        
        # Process data rows
        line_count = 0
        for line in infile:
            line = line.strip()
            
            # Skip empty lines
            if not line:
                continue
            
            # Parse the line - format: "HH:MM:SS AM/PM    CPU    %user    %nice    %system    %iowait    %steal    %idle"
            # Use regex to split on multiple spaces
            parts = re.split(r'\s+', line)
            
            if len(parts) < 9:
                print(f"parse_partition_io_log(): Warning - Skipping malformed line: {line}")
                continue
            
            timestamp = f"{parts[0]} {parts[1]}"  # Time and AM/PM
            cpu = parts[2]
            user_pct = parts[3]
            nice_pct = parts[4]
            system_pct = parts[5]
            iowait_pct = parts[6]
            steal_pct = parts[7]
            idle_pct = parts[8]
            
            # Write row to CSV
            csv_writer.writerow([timestamp, cpu, user_pct, nice_pct, system_pct, iowait_pct, steal_pct, idle_pct])
            line_count += 1
        
        print(f"parse_partition_io_log(): Successfully converted {line_count} rows to CSV")


if __name__ == '__main__':
    # Default input and output paths
    default_input = 'sar_logs_20251112_110805/partition_io.log'
    default_output = 'partition_io.csv'
    
    # Allow command line arguments for flexibility
    input_file = sys.argv[1] if len(sys.argv) > 1 else default_input
    output_file = sys.argv[2] if len(sys.argv) > 2 else default_output
    
    parse_partition_io_log(input_file, output_file)
    print(f"parse_partition_io_log(): Output written to: {output_file}")

