#!/usr/bin/env python3
"""
Convert process_creation.log file to CSV format.
Each timestamp becomes one row in the CSV output.
"""

import csv
import re
import sys
from pathlib import Path


def parse_process_creation_log(input_file, output_file):
    """
    Parse process_creation.log file and convert to CSV format.
    
    Args:
        input_file: Path to the input process_creation.log file
        output_file: Path to the output CSV file
    """
    input_path = Path(input_file)
    
    if not input_path.exists():
        print(f"parse_process_creation_log(): Error - Input file not found: {input_file}")
        sys.exit(1)
    
    # CSV column headers
    csv_headers = ['timestamp', 'dentunusd', 'file_nr', 'inode_nr', 'pty_nr']
    
    with open(input_path, 'r', encoding='utf-8') as infile, \
         open(output_file, 'w', newline='', encoding='utf-8') as outfile:
        
        csv_writer = csv.writer(outfile)
        csv_writer.writerow(csv_headers)
        
        # Skip the first 3 lines (system info header, empty line, column header)
        for _ in range(3):
            infile.readline()
        
        # Process data rows
        line_count = 0
        for line in infile:
            line = line.strip()
            
            # Skip empty lines
            if not line:
                continue
            
            # Parse the line - format: "HH:MM:SS AM/PM    dentunusd    file-nr    inode-nr    pty-nr"
            # Use regex to split on multiple spaces
            parts = re.split(r'\s+', line)
            
            if len(parts) < 6:
                print(f"parse_process_creation_log(): Warning - Skipping malformed line: {line}")
                continue
            
            timestamp = f"{parts[0]} {parts[1]}"  # Time and AM/PM
            dentunusd = parts[2]
            file_nr = parts[3]
            inode_nr = parts[4]
            pty_nr = parts[5]
            
            # Write row to CSV
            csv_writer.writerow([timestamp, dentunusd, file_nr, inode_nr, pty_nr])
            line_count += 1
        
        print(f"parse_process_creation_log(): Successfully converted {line_count} rows to CSV")


if __name__ == '__main__':
    # Default input and output paths
    default_input = 'sar_logs_20251112_110805/process_creation.log'
    default_output = 'process_creation.csv'
    
    # Allow command line arguments for flexibility
    input_file = sys.argv[1] if len(sys.argv) > 1 else default_input
    output_file = sys.argv[2] if len(sys.argv) > 2 else default_output
    
    parse_process_creation_log(input_file, output_file)
    print(f"parse_process_creation_log(): Output written to: {output_file}")

