#!/usr/bin/env python3
"""
Master script to convert all SAR log files to CSV format.
This script should be placed in the sar_log_* folder along with the log files.
It expects converter scripts to be in a 'converter_scripts' subfolder.
CSV output will be written to a 'CSV' subfolder.
"""

import os
import sys
from pathlib import Path

# Add converter_scripts to the path so we can import the converter modules
converter_scripts_dir = Path(__file__).parent / 'converter_scripts'
if converter_scripts_dir.exists():
    sys.path.insert(0, str(converter_scripts_dir))

# Import all converter functions
from convert_block_io import parse_block_io_log
from convert_context_switches import parse_context_switches_log
from convert_cpu_usage import parse_cpu_usage_log
from convert_disk_io import parse_disk_io_log
from convert_ip_stats import parse_ip_stats_log
from convert_load_avg import parse_load_avg_log
from convert_mem_paging import parse_mem_paging_log
from convert_mem_usage import parse_mem_usage_log
from convert_net_dev import parse_net_dev_log
from convert_net_errors import parse_net_errors_log
from convert_page_faults import parse_page_faults_log
from convert_paging import parse_paging_log
from convert_partition_io import parse_partition_io_log
from convert_per_cpu import parse_per_cpu_log
from convert_process_creation import parse_process_creation_log
from convert_sock_usage import parse_sock_usage_log
from convert_swap_usage import parse_swap_usage_log
from convert_tcp_udp import parse_tcp_udp_log
from convert_tty_activity import parse_tty_activity_log


def convert_all_logs(input_dir='.', output_dir='CSV'):
    """
    Convert all SAR log files to CSV format.
    
    Args:
        input_dir: Directory containing the log files (default: current directory)
        output_dir: Directory to write CSV files (default: CSV subfolder)
    """
    input_path = Path(input_dir)
    output_path = Path(output_dir)
    
    # Verify converter_scripts directory exists
    converter_scripts_path = Path(__file__).parent / 'converter_scripts'
    if not converter_scripts_path.exists():
        print("convert_all_logs(): Error - converter_scripts folder not found!")
        print(f"convert_all_logs(): Expected location: {converter_scripts_path.absolute()}")
        print("convert_all_logs(): Please ensure converter_scripts folder exists in the same directory as this script.")
        sys.exit(1)
    
    # Verify log files exist in input directory
    if not (input_path / 'block_io.log').exists():
        print("convert_all_logs(): Error - Could not find log files in current directory.")
        print(f"convert_all_logs(): Looking in: {input_path.absolute()}")
        print("convert_all_logs(): Please ensure you are running this script from the sar_log_* folder.")
        sys.exit(1)
    
    # Create output directory if it doesn't exist
    output_path.mkdir(exist_ok=True)
    print(f"convert_all_logs(): Input directory: {input_path.absolute()}")
    print(f"convert_all_logs(): Output directory: {output_path.absolute()}")
    
    # Define all log file conversions
    conversions = [
        ('block_io.log', parse_block_io_log, 'block_io.csv'),
        ('context_switches.log', parse_context_switches_log, 'context_switches.csv'),
        ('cpu_usage.log', parse_cpu_usage_log, 'cpu_usage.csv'),
        ('disk_io.log', parse_disk_io_log, 'disk_io.csv'),
        ('ip_stats.log', parse_ip_stats_log, 'ip_stats.csv'),
        ('load_avg.log', parse_load_avg_log, 'load_avg.csv'),
        ('mem_paging.log', parse_mem_paging_log, 'mem_paging.csv'),
        ('mem_usage.log', parse_mem_usage_log, 'mem_usage.csv'),
        ('net_dev.log', parse_net_dev_log, 'net_dev.csv'),
        ('net_errors.log', parse_net_errors_log, 'net_errors.csv'),
        ('page_faults.log', parse_page_faults_log, 'page_faults.csv'),
        ('paging.log', parse_paging_log, 'paging.csv'),
        ('partition_io.log', parse_partition_io_log, 'partition_io.csv'),
        ('per_cpu.log', parse_per_cpu_log, 'per_cpu.csv'),
        ('process_creation.log', parse_process_creation_log, 'process_creation.csv'),
        ('sock_usage.log', parse_sock_usage_log, 'sock_usage.csv'),
        ('swap_usage.log', parse_swap_usage_log, 'swap_usage.csv'),
        ('tcp_udp.log', parse_tcp_udp_log, 'tcp_udp.csv'),
        ('tty_activity.log', parse_tty_activity_log, 'tty_activity.csv'),
    ]
    
    successful = 0
    failed = 0
    skipped = 0
    
    print(f"\nconvert_all_logs(): Starting conversion of log files...")
    print(f"convert_all_logs(): Looking for log files in: {input_path.absolute()}\n")
    
    for log_filename, parse_func, csv_filename in conversions:
        log_path = input_path / log_filename
        csv_path = output_path / csv_filename
        
        if not log_path.exists():
            print(f"convert_all_logs(): Skipping {log_filename} - file not found")
            skipped += 1
            continue
        
        try:
            print(f"convert_all_logs(): Converting {log_filename}...")
            parse_func(str(log_path), str(csv_path))
            successful += 1
        except Exception as e:
            print(f"convert_all_logs(): ERROR converting {log_filename}: {e}")
            failed += 1
    
    print(f"\nconvert_all_logs(): Conversion complete!")
    print(f"convert_all_logs(): Successful: {successful}")
    print(f"convert_all_logs(): Failed: {failed}")
    print(f"convert_all_logs(): Skipped: {skipped}")
    print(f"convert_all_logs(): CSV files written to: {output_path.absolute()}")


if __name__ == '__main__':
    # Allow command line arguments for input and output directories
    input_dir = sys.argv[1] if len(sys.argv) > 1 else '.'
    output_dir = sys.argv[2] if len(sys.argv) > 2 else 'CSV'
    
    convert_all_logs(input_dir, output_dir)
